/*==================================================
  tabbedComments.js by Patrick Fitzgerald pat@barelyfitz.com
  http://www.barelyfitz.com/projects/tabbedComments/

  License (http://www.opensource.org/licenses/mit-license.php)

  Copyright (c) 2006 Patrick Fitzgerald

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation files
  (the "Software"), to deal in the Software without restriction,
  including without limitation the rights to use, copy, modify, merge,
  publish, distribute, sublicense, and/or sell copies of the Software,
  and to permit persons to whom the Software is furnished to do so,
  subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
  ==================================================

  Contains three functions:

  tabbedCommentsObj()
  Constructor for a tabbedComments object

  tabbedCommentsAutomatic()
  Searches the document for elements to convert to tabbedComments interfaces

  tabbedCommentsAutomaticOnLoad()
  Adds an onload event to the document to call tabbedCommentsAutomatic

  ==================================================*/

function tabbedCommentsObj()
{
  /* This object converts the contents of a div element
     into a dynamic tabbed interface.

     Example:
     var mytab = new tabbedCommentsObj();
     mytab.init(document.getElementById('mydiv'));
  */

  /*--------------------------------------------------
    Properties
    --------------------------------------------------*/

  /* Class of the main tabbedComments div */
  this.classMain = "tabbedComments";

  /* Rename classMain to classMainLive after tabifying
     (so a different style can be applied)
  */
  this.classMainLive = "tabbedCommentslive";

  /* Class of each DIV that contains a tab */
  this.classTab = "commentTab";

  /* Class for the navigation UL */
  this.classNav = "tabbedCommentsnav";

  /* When a tab is to be hidden, instead of setting display='none', we
     set the class of the div to classTabHide. In your screen
     stylesheet you should set classTabHide to display:none.  In your
     print stylesheet you should set display:block to ensure that all
     the information is printed.
  */
  this.classTabHide = "commentTabhide";

  /* Class to set the navigation LI when the tab is active, so you can
     use a different style on the active tab.
  */
  this.classNavActive = "tabbedCommentsactive";

  /* Array of objects holding info about each tab */
  this.tabs = new Array();
}

/*--------------------------------------------------
  Methods for tabbedCommentsObj
  --------------------------------------------------*/

tabbedCommentsObj.prototype.init = function(e)
{
  /* Set up the tabbedComments interface.

     e = element (the main containing div)

     Example:
     init(document.getElementById('mytabbedCommentsdiv'))
   */

  var i;

  /* Attach this object to the tabbedComments div so anyone can reference it. */
  e.tabbedComments = this;

  /* Verify that the browser supports DOM scripting */
  if (!document.getElementsByTagName) { return false; }


  /* Clear the existing tabs array (shouldn't be necessary) */
  this.tabs.length = 0;

  /* Create some regular expressions to match class names */
  var reMain = new RegExp('\\b' + this.classMain + '\\b', 'i');
  var reTab = new RegExp('\\b' + this.classTab + '\\b', 'i');

  /* Loop through an array of all the child nodes within our tabbedComments element. */
  var childNodes = e.childNodes;
  for(i=0; i < childNodes.length; i++) {

    /* Find the nodes where class="commentTab" */
    if(childNodes[i].className && childNodes[i].className.match(reTab)) {
      
      /* Create a new object to save info about this tab */
      t = new Object();
      
      /* Save a pointer to the div for this tab */
      t.div = childNodes[i];
      
      /* Add the new object to the array of tabs */
      this.tabs[this.tabs.length] = t;
    }
  }

  /* Create a new UL list to hold the tab headings */
  var ul = document.createElement("ul");
  ul.className = this.classNav;
  
  /* Loop through each tab we found */
  for (i=0; i < this.tabs.length; i++) {

    var t = this.tabs[i];

    /* Get the heading from the title attribute on the DIV,
       or just use an automatically generated number.
    */
    t.headingText = t.div.title;
    if (!t.headingText) {
      t.headingText = i + 1;
    }

    /* Create a link to activate the tab */
    t.a = document.createElement("a");
    t.a.appendChild(document.createTextNode(""));
    t.a.firstChild.data = t.headingText;
    t.a.href = "javascript:void(null);";
    t.a.title = t.headingText;
    t.a.onclick = this.onClick;

    /* Add some properties to the link so we can identify which tab
       was clicked. Later the onClick method will need this.
    */
    t.a.tabbedComments = this;
    t.a.commentTabid = i;

    /* Create a list element for the tab */
    t.li = document.createElement("li");

    /* Add the link to the list element */
    t.li.appendChild(t.a);

    /* Add the list element to the list */
    ul.appendChild(t.li);
  }

  /* Add the UL list to the tabbedComments */
  e.insertBefore(ul, e.firstChild);

  /* Replace the class so a different style can be applied */
  e.className = e.className.replace(reMain, this.classMainLive);

  /* Activate the first tab */
  this.tabShow(0);

  return this;
}

tabbedCommentsObj.prototype.onClick = function()
{
  /* This method should only be called by the onClick event of an <A>
     element, in which case we will determine which tab was clicked by
     examining a property that we previously attached to the <A>
     element.

     Since this was triggered from an onClick event, the variable
     "this" refers to the <A> element that triggered the onClick
     event (and not to the tabbedCommentsObj).

     When tabbedCommentsObj was initialized, we added some extra properties
     to the <A> element, for the purpose of retrieving them now. Get
     the tabbedCommentsObj object, plus the tab number that was clicked.
  */
  var a = this;
  if (!a.tabbedComments) { return false; }

  var self = a.tabbedComments;
  var tabindex = a.commentTabid;

  /* Remove focus from the link because it looks ugly.
     I don't know if this is a good idea...
  */
  a.blur();

  self.tabShow(tabindex);

  return false;
}

tabbedCommentsObj.prototype.tabHideAll = function()
{
  /* Hide all tabs and make all navigation links inactive */
  for (var i = 0; i < this.tabs.length; i++) {
    this.tabHide(i);
  }
}

tabbedCommentsObj.prototype.tabHide = function(tabindex)
{
  /* Hide the tab and make its navigation link inactive */
  var div = this.tabs[tabindex].div;

  /* Hide the tab contents by adding classTabHide to the div */
  var re = new RegExp('\\b' + this.classTabHide + '\\b');
  if (!div.className.match(re)) {
    div.className += ' ' + this.classTabHide;
  }
  this.navClearActive(tabindex);
}

tabbedCommentsObj.prototype.tabShow = function(tabindex)
{
  /* Note: this method enforces the rule that only one tab at a time
     should be active.
  */

  if (!this.tabs[tabindex]) { return false; };

  /* Hide all the tabs first */
  this.tabHideAll();

  /* Get the div that holds this tab */
  var div = this.tabs[tabindex].div;

  /* Remove classTabHide from the div */
  var re = new RegExp('\\b' + this.classTabHide + '\\b', 'g');
  div.className = div.className.replace(re, '');

  /* Mark this tab navigation link as "active" */
  this.navSetActive(tabindex);

  return this;
}

tabbedCommentsObj.prototype.navSetActive = function(tabindex)
{
  /* Note: this method does *not* enforce the rule
     that only one nav item can be active at a time.
  */

  /* Set classNavActive for the navigation list item */
  this.tabs[tabindex].li.className = this.classNavActive;

  return this;
}

tabbedCommentsObj.prototype.navClearActive = function(tabindex)
{
  /* Note: this method does *not* enforce the rule
     that one nav should always be active.
  */

  /* Remove classNavActive from the navigation list item */
  this.tabs[tabindex].li.className = '';

  return this;
}

/*==================================================*/

function tabbedCommentsAutomatic()
{
  /* This function finds all DIV elements in the document where class=tabbedComments,
     then converts them to use the tabbedComments interface */

  /* Verify that the browser supports DOM scripting */
  if (!document.getElementsByTagName) { return false; }

  /* Create a tabbedComments object so we can get the value of classMain
     (so we don't have to duplicate the value here and cause confusion)
  */
  var tempObj = new tabbedCommentsObj();
  
  /* Find all DIV elements in the document that have class=tabbedComments */

  /* First get an array of all DIV elements and loop through them */
  var divs = document.getElementsByTagName("div");
  for (var i=0; i < divs.length; i++) {
    
    /* Find class=tabbedComments */
    if(divs[i].className == tempObj.classMain) {
      
      /* Now tabify it - attach the tabbedCommentsObj directly to the div element
	 so we don't have to use a global variable */
      divs[i].tabbedComments = new tabbedCommentsObj();
      divs[i].tabbedComments.init(divs[i]);
    }
  }
  
  return this;
}

/*==================================================*/

function tabbedCommentsAutomaticOnLoad()
{
  /* This function adds tabbedCommentsAutomatic to the window.onload event,
     so it will run after the document has finished loading.
  */

  /* Taken from: http://simon.incutio.com/archive/2004/05/26/addLoadEvent */

  var oldonload = window.onload;
  if (typeof window.onload != 'function') {
    window.onload = tabbedCommentsAutomatic;
  } else {
    window.onload = function() {
      oldonload();
      tabbedCommentsAutomatic();
    }
  }
}

/*==================================================*/

/* Run tabbedCommentsAutomaticOnload() */
/* Remove the following line if you don't want tabbedComments to run
   automatically when the script is loaded */
tabbedCommentsAutomaticOnLoad();